/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Reflection;
using System.Collections;

namespace Borland.Eco.Persistence.Configuration
{
	///<summary>
	///Enumeration passed to DatabaseEngineInitializer.InitializeDbEngineSettings to quickly set the
	///correct properties for an engine.
	///</summary>
	public enum  DataBaseEngine
	{
		Unknown,
		InterbaseSqlDialect1, InterbaseSqlDialect3,
		GenericANSISQL92,
		SqlServerBdp,
		SqlServerSqlClient,
		SqlServer7Bdp,
		SqlServer7SqlClient,
		DBISAM,
		Oracle,
		Advantage,
		Informix,
		DB2,
		SqlServerMBCSBdp,
		SqlServerMBCSSqlClient,
		SqlServer7MBCSBdp,
		SqlServer7MBCSSqlClient,
		OracleMBCS,
		InterbaseSqlDialect3MBCS,
		DB2MBCS,
		SybaseCommon,
		SybaseBdp,
		AccessBdp,
		AccessSqlClient,
		NDataStore,
		NDataStoreBdp,
		MySqlBdp
	};

	public sealed class StringLiterals
	{
		private StringLiterals() {}
		public const string SystemDateTime = "System.DateTime";
		public const string SystemByteArr = "System.Byte[]";
		public const string SystemString = "System.String";
		public const string SystemBoolean = "System.Boolean";
		public const string SystemByte = "System.Byte";
		public const string SystemSByte = "System.SByte";
		public const string SystemInt16 = "System.Int16";
		public const string SystemUInt16 = "System.UInt16";
		public const string SystemInt32 = "System.Int32";
		public const string SystemUInt32 = "System.UInt32";
		public const string SystemInt64 = "System.Int64";
		public const string SystemUInt64 = "System.UInt64";
		public const string SystemDouble = "System.Double";
		public const string SystemSingle = "System.Single";
		public const string SystemDecimal = "System.Decimal";
		public const string SystemChar = "System.Char";
		public const string SystemGuid = "System.Guid";
		public const string SystemTimeSpan = "System.TimeSpan";
		public const string AutoInc = "AutoInc";
		public const string AutoInc16 = "AutoInc16";
		public const string AutoIncDecimal = "AutoIncDecimal";
		public const string Guid = "Guid";
		public const string AutoGuid = "AutoGuid";
		public const string Attribute = "Attribute";
		public const string GuidAsVarChar32 = "GuidAsVarChar32";
		
		public const string Persistence = "Borland.Eco.Persistence";
		public const string PersistenceBdp = "Borland.Eco.Persistence.Bdp";
		public const string PersistenceSql = "Borland.Eco.Persistence.SqlServer";
		public const string PersistenceNDataStore = "Borland.Eco.Persistence.NDataStore";
		public const string Interfaces = "Borland.Eco.Interfaces";
	}

	///<summary>
	///<para>Primarily intended for use by designers.</para>
	///<para>Can also be used to switch database engine in run time.</para>
	///</summary>
	public sealed class DatabaseEngineInitializer
	{
		private DatabaseEngineInitializer() {}
		
		private static Hashtable assemblies = new Hashtable();
		private static Assembly GetAssembly(string assemblyName)
		{
			Assembly result = (Assembly)assemblies[assemblyName];
			if (result == null)
			{
#if CF
				result = Assembly.LoadFrom(assemblyName + ".dll");
#else
				result = Assembly.LoadWithPartialName(assemblyName);
#endif
				assemblies.Add(assemblyName, result);
			}
			return result;
		}
		private static Type RetrieveType(string assemblyName, string typeName)
		{
			if (assemblyName == null || assemblyName.Length == 0)
				throw new ArgumentNullException("assemblyName"); // do not localize

			Assembly a = GetAssembly(assemblyName);

			if (a == null)
				throw new InvalidOperationException(PersistenceStringRes.sCouldNotLoadAssembly(assemblyName));

			if (typeName == String.Empty || typeName == null)
				throw new ArgumentNullException("typeName"); // do not localize

			return a.GetType(typeName, true); // Throw exception if type not found
		}

		private static void SetInitialValues(SqlDatabaseConfig cfg)
		{
			cfg.SetInitialValuesForBasicProperties();
			cfg.PersistenceMappers.Clear();
			cfg.SetPMapper(StringLiterals.SystemChar, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.CharAsChar")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemString, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.StringAsVarChar")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDateTime, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.DateTimeAsTimeStamp")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemGuid, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.GuidAsVarChar32")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemBoolean, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.BooleanAsInteger")); // do not localize
			cfg.SetPMapper("BooleanAsYN", RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.BooleanAsYNChar")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemByte, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.ByteAsSmallInt")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemSByte, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.SByteAsSmallInt")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt16, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.Int16AsSmallint")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemUInt16, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.UInt16AsLongInteger")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt32, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.Int32AsInteger")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemUInt32, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.UInt32AsLongInteger")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt64, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.Int64AsLongInteger")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemUInt64, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.UInt64AsDecimal")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemSingle, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.SingleAsSingle")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDouble, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.DoubleAsDouble")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.DecimalAsDecimal")); // do not localize
			cfg.SetKeyMapper(DefaultEcoIdMapper.DefaultKeyMapperName, RetrieveType(StringLiterals.Interfaces, "Borland.Eco.Persistence.DefaultEcoIdMapper")); // do not localize
			cfg.SetKeyMapper(StringLiterals.Attribute, RetrieveType(StringLiterals.Interfaces, "Borland.Eco.Persistence.AttributeKeyMapper")); // do not localize
			cfg.SetKeyMapper(StringLiterals.AutoInc, RetrieveType(StringLiterals.Interfaces, "Borland.Eco.Persistence.AutoIncKeyMapper")); // do not localize
			cfg.SetKeyMapper(StringLiterals.Guid, RetrieveType(StringLiterals.Interfaces, "Borland.Eco.Persistence.GuidKeyMapper")); // do not localize
		}

#region Initialization for DB engines
		private static void InitializeSQLServerCommon(SqlDatabaseConfig cfg)
		{
			/*	fColumnTypeForDate := 'DATETIME';
			fColumnTypeForTime := 'DATETIME';
			fColumnTypeForDateTime := 'DATETIME';
			fColumnTypeForFloat := 'DECIMAL (28,10)';
			fColumnTypeForCurrency := 'DECIMAL (28,10)';
			*/
			cfg.DateTimeFormat = "\"'\"yyyyMMdd\"'\""; // do not localize
			cfg.UseSQL92Joins = true;

			cfg.DropColumnTemplate = "DECLARE @defname VARCHAR(100), @cmd VARCHAR(1000) SET @defname = (SELECT name FROM sysobjects so JOIN sysconstraints sc ON so.id = sc.constid  WHERE object_name(so.parent_obj) = '<TableName>' AND so.xtype = 'D' AND sc.colid = (SELECT colid FROM syscolumns WHERE id = object_id('dbo.<TableName>') AND name = '<ColumnName>')) SET @cmd = 'ALTER TABLE <TableName> DROP CONSTRAINT ' + @defname EXEC(@cmd);ALTER TABLE <TableName> DROP COLUMN <ColumnName>"; // do not localize
			cfg.DropIndexTemplate = "DROP INDEX <TableName>.<IndexName>"; // do not localize
			cfg.BooleanTrueLiteral = "1";
			cfg.LengthFunctionName = "LEN";
			cfg.SetPMapper(StringLiterals.SystemDateTime, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.DateTimeAsDateTime")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemTimeSpan, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.TimeSpanAsDateTime")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.DecimalAsDecimal")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemSingle, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.SingleAsReal")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt64, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.Int64AsLongInteger")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoInc, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.Int32AsAutoInc")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoInc16, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.Int16AsAutoInc")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoIncDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.DecimalAsAutoInc")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemByte, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.ByteAsTinyInt")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemBoolean, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.BooleanAsBit")); // do not localize
			cfg.SetPMapper("BooleanAsInteger", RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.BooleanAsInteger")); // do not localize
		}

		private static void InitializeSQLServerBdp(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerCommon(cfg);
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.SqlServer.Bdp.ByteArrayAsImage")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemByte, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.SqlServer.Bdp.ByteAsTinyInt")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.SqlServer.Bdp.StringAsText")); // do not localize
			cfg.SetPMapper("StringAsNText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.SqlServer.Bdp.StringAsNText")); // do not localize
		}

		private static void InitializeAccessCommon(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerCommon(cfg);
			cfg.DateTimeFormat = "";
			cfg.UpperCaseFunctionName = "UCASE"; // do not localize
			cfg.DropIndexTemplate = "DROP INDEX <IndexName> ON <TableName>"; // do not localize
		}

		private static void InitializeAccessBdp(SqlDatabaseConfig cfg)
		{
			InitializeAccessCommon(cfg);
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.SqlServer.Bdp.ByteArrayAsImage")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemByte, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.SqlServer.Bdp.ByteAsTinyInt")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.SqlServer.Bdp.StringAsText")); // do not localize
			cfg.SetPMapper("StringAsNText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.SqlServer.Bdp.StringAsNText")); // do not localize
		}

		private static void InitializeAccessSqlClient(SqlDatabaseConfig cfg)
		{
			InitializeAccessCommon(cfg);
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceSql, "Borland.Eco.Persistence.SqlServer.SqlClient.ByteArrayAsImage")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceSql, "Borland.Eco.Persistence.SqlServer.SqlClient.StringAsText")); // do not localize
			cfg.SetPMapper("StringAsNText", RetrieveType(StringLiterals.PersistenceSql, "Borland.Eco.Persistence.SqlServer.SqlClient.StringAsNText")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemGuid, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.GuidAsUniqueIdentifier")); // do not localize
			cfg.SetPMapper(StringLiterals.GuidAsVarChar32, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.GuidAsVarChar32")); // do not localize
		}

		private static void InitializeSQLServerSqlClient(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerCommon(cfg);
			cfg.IdentifierCharacterMode = IdentifierCharacterMode.AllowUnicode;
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceSql, "Borland.Eco.Persistence.SqlServer.SqlClient.ByteArrayAsImage")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceSql, "Borland.Eco.Persistence.SqlServer.SqlClient.StringAsText")); // do not localize
			cfg.SetPMapper("StringAsNText", RetrieveType(StringLiterals.PersistenceSql, "Borland.Eco.Persistence.SqlServer.SqlClient.StringAsNText")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemGuid, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.GuidAsUniqueIdentifier")); // do not localize
			cfg.SetPMapper(StringLiterals.GuidAsVarChar32, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.GuidAsVarChar32")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoGuid, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.GuidAsUniqueIdentifierWithDefault")); // do not localize
			cfg.SetKeyMapper(StringLiterals.AutoGuid, RetrieveType(StringLiterals.Interfaces, "Borland.Eco.Persistence.AutoGuidKeyMapper")); // do not localize
		}

		private static void InitializeSQLServerMBCSBdp(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerBdp(cfg);
			cfg.SetPMapper(StringLiterals.SystemString, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.StringAsNVarChar")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemChar, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.CharAsNChar")); // do not localize
		}

		private static void InitializeSQLServerMBCSSqlClient(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerSqlClient(cfg);
			cfg.SetPMapper(StringLiterals.SystemString, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.StringAsNVarChar")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemChar, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.CharAsNChar")); // do not localize
		}
		
		private static void InitializeSQLServer7Bdp(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerBdp(cfg);
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.DecimalAsDecimal28")); // do not localize
		}

		private static void InitializeSQLServer7SqlClient(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerSqlClient(cfg);
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.DecimalAsDecimal28")); // do not localize
		}

		private static void InitializeSQLServer7MBCSBdp(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerMBCSBdp(cfg);
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.DecimalAsDecimal28")); // do not localize
		}

		private static void InitializeSQLServer7MBCSSqlClient(SqlDatabaseConfig cfg)
		{
			InitializeSQLServerMBCSSqlClient(cfg);
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.DecimalAsDecimal28")); // do not localize
		}


		private static void InitializeGenericANSISQL92(SqlDatabaseConfig cfg)
		{
				/*
				fColumnTypeForDate := 'DATE';
				fColumnTypeForTime := 'TIME';
				fColumnTypeForDateTime := 'TIMESTAMP';
				*/
		}

		private static void InitializeInterbaseSQLDialect3(SqlDatabaseConfig cfg)
		{
			/*
			fColumnTypeForDate := 'TIMESTAMP';
			fColumnTypeForTime := 'TIMESTAMP';
			fColumnTypeForDateTime := 'TIMESTAMP';
			*/
			cfg.MaxIndexNameLength = 31;
			cfg.MaxDbIdentifierLength = 31;
			cfg.UseSQL92Joins = true;
			cfg.AllowMetadataChangesInTransaction = true;
			cfg.DateTimeFormat = "\"'\"yyyy-MM-dd\"'\""; // do not localize
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Interbase.ByteArrayAsBdpBlob")); // do not localize
			cfg.SetPMapper("StringAsBlob", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Interbase.StringAsBdpBlob")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Interbase.StringAsBdpBlob")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemTimeSpan, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.TimeSpanAsTimeStamp")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Interbase.DecimalAsDecimal")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt64, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Interbase.Int64AsLongInteger")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoInc, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Interbase.Int32AsAutoInc")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoInc16, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Interbase.Int16AsAutoInc")); // do not localize
			cfg.SetKeyMapper(StringLiterals.AutoInc16, RetrieveType(StringLiterals.Interfaces, "Borland.Eco.Persistence.AutoInc16KeyMapper")); // do not localize

			cfg.AddReservedWord("ACTIVE"); // do not localize
			cfg.AddReservedWord("ADMIN"); // do not localize
			cfg.AddReservedWord("AFTER"); // do not localize
			cfg.AddReservedWord("ASCENDING"); // do not localize
			cfg.AddReservedWord("AUTO"); // do not localize
			cfg.AddReservedWord("AUTODDL"); // do not localize
			cfg.AddReservedWord("BASED"); // do not localize
			cfg.AddReservedWord("BASENAME"); // do not localize
			cfg.AddReservedWord("BASE_NAME"); // do not localize
			cfg.AddReservedWord("BEFORE"); // do not localize
			cfg.AddReservedWord("BLOB"); // do not localize
			cfg.AddReservedWord("BLOBEDIT"); // do not localize
			cfg.AddReservedWord("BUFFER"); // do not localize
			cfg.AddReservedWord("CACHE"); // do not localize
			cfg.AddReservedWord("CHECK_POINT_LEN"); // do not localize
			cfg.AddReservedWord("CHECK_POINT_LENGTH"); // do not localize
			cfg.AddReservedWord("COMMITTED"); // do not localize
			cfg.AddReservedWord("COMPILETIME"); // do not localize
			cfg.AddReservedWord("COMPUTED"); // do not localize
			cfg.AddReservedWord("CONDITIONAL"); // do not localize
			cfg.AddReservedWord("CONTAINING"); // do not localize
			cfg.AddReservedWord("CSTRING"); // do not localize
			cfg.AddReservedWord("DATABASE"); // do not localize
			cfg.AddReservedWord("DB_KEY"); // do not localize
			cfg.AddReservedWord("DEBUG"); // do not localize
			cfg.AddReservedWord("DESCENDING"); // do not localize
			cfg.AddReservedWord("DISPLAY"); // do not localize
			cfg.AddReservedWord("DO"); // do not localize
			cfg.AddReservedWord("ECHO"); // do not localize
			cfg.AddReservedWord("EDIT"); // do not localize
			cfg.AddReservedWord("ENTRY_POINT"); // do not localize
			cfg.AddReservedWord("EVENT"); // do not localize
			cfg.AddReservedWord("EXIT"); // do not localize
			cfg.AddReservedWord("EXTERN"); // do not localize
			cfg.AddReservedWord("FILE"); // do not localize
			cfg.AddReservedWord("FILTER"); // do not localize
			cfg.AddReservedWord("FREE_IT"); // do not localize
			cfg.AddReservedWord("FUNCTION"); // do not localize
			cfg.AddReservedWord("GDSCODE"); // do not localize
			cfg.AddReservedWord("GENERATOR"); // do not localize
			cfg.AddReservedWord("GEN_ID"); // do not localize
			cfg.AddReservedWord("GROUP_COMMIT_WAIT"); // do not localize
			cfg.AddReservedWord("GROUP_COMMIT_WAIT_TIME"); // do not localize
			cfg.AddReservedWord("HELP"); // do not localize
			cfg.AddReservedWord("IF"); // do not localize
			cfg.AddReservedWord("INACTIVE"); // do not localize
			cfg.AddReservedWord("INDEX"); // do not localize
			cfg.AddReservedWord("INIT"); // do not localize
			cfg.AddReservedWord("INPUT_TYPE"); // do not localize
			cfg.AddReservedWord("ISQL"); // do not localize
			cfg.AddReservedWord("LC_MESSAGES"); // do not localize
			cfg.AddReservedWord("LC_TYPE"); // do not localize
			cfg.AddReservedWord("LENGTH"); // do not localize
			cfg.AddReservedWord("LEV"); // do not localize
			cfg.AddReservedWord("LOGFILE"); // do not localize
			cfg.AddReservedWord("LOG_BUFFER_SIZE"); // do not localize
			cfg.AddReservedWord("LOG_BUF_SIZE"); // do not localize
			cfg.AddReservedWord("LONG"); // do not localize
			cfg.AddReservedWord("MANUAL"); // do not localize
			cfg.AddReservedWord("MAXIMUM"); // do not localize
			cfg.AddReservedWord("MAXIMUM_SEGMENT"); // do not localize
			cfg.AddReservedWord("MAX_SEGMENT"); // do not localize
			cfg.AddReservedWord("MERGE"); // do not localize
			cfg.AddReservedWord("MESSAGE"); // do not localize
			cfg.AddReservedWord("MINIMUM"); // do not localize
			cfg.AddReservedWord("MODULE_NAME"); // do not localize
			cfg.AddReservedWord("NOAUTO"); // do not localize
			cfg.AddReservedWord("NUM_LOG_BUFFERS"); // do not localize
			cfg.AddReservedWord("NUM_LOG_BUFS"); // do not localize
			cfg.AddReservedWord("OUTPUT_TYPE"); // do not localize
			cfg.AddReservedWord("OVERFLOW"); // do not localize
			cfg.AddReservedWord("PAGE"); // do not localize
			cfg.AddReservedWord("PAGELENGTH"); // do not localize
			cfg.AddReservedWord("PAGES"); // do not localize
			cfg.AddReservedWord("PAGE_SIZE"); // do not localize
			cfg.AddReservedWord("PARAMETER"); // do not localize
			cfg.AddReservedWord("PASSWORD"); // do not localize
			cfg.AddReservedWord("PLAN"); // do not localize
			cfg.AddReservedWord("POST_EVENT"); // do not localize
			cfg.AddReservedWord("QUIT"); // do not localize
			cfg.AddReservedWord("RAW_PARTITIONS"); // do not localize
			cfg.AddReservedWord("RDB$DB_KEY"); // do not localize
			cfg.AddReservedWord("RECORD_VERSION"); // do not localize
			cfg.AddReservedWord("RELEASE"); // do not localize
			cfg.AddReservedWord("RESERV"); // do not localize
			cfg.AddReservedWord("RESERVING"); // do not localize
			cfg.AddReservedWord("RETAIN"); // do not localize
			cfg.AddReservedWord("RETURN"); // do not localize
			cfg.AddReservedWord("RETURNING_VALUES"); // do not localize
			cfg.AddReservedWord("RETURNS"); // do not localize
			cfg.AddReservedWord("ROLE"); // do not localize
			cfg.AddReservedWord("RUNTIME"); // do not localize
			cfg.AddReservedWord("SECOND"); // do not localize
			cfg.AddReservedWord("SHADOW"); // do not localize
			cfg.AddReservedWord("SHARED"); // do not localize
			cfg.AddReservedWord("SHELL"); // do not localize
			cfg.AddReservedWord("SHOW"); // do not localize
			cfg.AddReservedWord("SINGULAR"); // do not localize
			cfg.AddReservedWord("SNAPSHOT"); // do not localize
			cfg.AddReservedWord("SORT"); // do not localize
			cfg.AddReservedWord("STABILITY"); // do not localize
			cfg.AddReservedWord("STARTING"); // do not localize
			cfg.AddReservedWord("STARTS"); // do not localize
			cfg.AddReservedWord("STATEMENT"); // do not localize
			cfg.AddReservedWord("STATIC"); // do not localize
			cfg.AddReservedWord("STATISTICS"); // do not localize
			cfg.AddReservedWord("SUB_TYPE"); // do not localize
			cfg.AddReservedWord("SUSPEND"); // do not localize
			cfg.AddReservedWord("TERMINATOR"); // do not localize
			cfg.AddReservedWord("TRIGGER"); // do not localize
			cfg.AddReservedWord("TYPE"); // do not localize
			cfg.AddReservedWord("UNCOMMITTED"); // do not localize
			cfg.AddReservedWord("VARIABLE"); // do not localize
			cfg.AddReservedWord("VERSION"); // do not localize
			cfg.AddReservedWord("WAIT"); // do not localize
			cfg.AddReservedWord("WEEKDAY"); // do not localize
			cfg.AddReservedWord("WHILE"); // do not localize
			cfg.AddReservedWord("YEARDAY"); // do not localize
		}

		private static void InitializeInterbaseSQLDialect3MBCS(SqlDatabaseConfig cfg)
		{
			InitializeInterbaseSQLDialect3(cfg);
			cfg.SetPMapper(StringLiterals.SystemString, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Interbase.StringAsNCharUnicode")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Interbase.StringAsBdpBlob")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemChar, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Interbase.CharAsNCharUnicode")); // do not localize
		}

		private static void InitializeInterbaseSQLDialect1(SqlDatabaseConfig cfg)
		{
			cfg.MaxIndexNameLength = 31;
			cfg.MaxDbIdentifierLength = 31;
			cfg.AllowMetadataChangesInTransaction = true;
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Interbase.ByteArrayAsBdpBlob")); // do not localize
			cfg.SetPMapper("StringAsBlob", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Interbase.StringAsBdpBlob")); // do not localize
		}

		private static void InitializeDBISAM(SqlDatabaseConfig cfg)
			{
				/*
				fColumnTypeForDate := 'DATE';
				fColumnTypeForTime := 'TIME';
				fColumnTypeForDateTime := 'TIMESTAMP';
				fColumnTypeForFloat := 'FLOAT';
				fColumnTypeForCurrency := 'FLOAT';
				*/
				cfg.DefaultStringLength = 250;
				//	   cfg.AddReservedWord("Description");
		}

		private static void InitializeAdvantage(SqlDatabaseConfig cfg)
			{
				cfg.SQLforNotNull = "CONSTRAINT NOT NULL"; // do not localize
				/*
				fColumnTypeForSmallInt := 'SHORT';
				fColumnTypeForFloat := 'NUMERIC';
				fColumnTypeForCurrency := 'NUMERIC';
				fSupportsConstraintsInCreateTable := false;
				fQuoteNonStringDefaultValues := true;
				fSupportsStringDefaultValues := false;
				fColumnTypeForDate := 'DATE';
				fColumnTypeForTime := 'TIME';
				fColumnTypeForDateTime := 'TIMESTAMP';
				fColumnTypeForString := 'CHAR(%d)';

				fStoreEmptyStringsAsNULL := True;
				*/
		}

		private static void InitializeSybaseCommon(SqlDatabaseConfig cfg)
		{
			cfg.AllowMetadataChangesInTransaction = false;
			cfg.MaxDbIdentifierLength = 30;
			cfg.MaxIndexNameLength = 30;
			cfg.SQLforAllowNull = "NULL";
			cfg.DropIndexTemplate = "DROP INDEX <TableName>.<IndexName>"; // do not localize
			cfg.SetPMapper(StringLiterals.SystemBoolean, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Sybase.BooleanAsBit")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemSingle, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Sybase.SingleAsFloat")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemTimeSpan, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.TimeSpanAsDateTime")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDateTime, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.DateTimeAsDateTime")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoInc, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Sybase.Int32AsNumericIdentity")); // do not localize
		}

		private static void InitializeSybaseBdp(SqlDatabaseConfig cfg)
		{
			InitializeSybaseCommon(cfg);
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Sybase.Bdp.ByteArrayAsImage")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Sybase.Bdp.StringAsText")); // do not localize
//			cfg.SetPMapper("StringAsNText", RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.Bdp.StringAsNText")); // do not localize
		}

/*
		private static void InitializeSybaseAse(SqlDatabaseConfig cfg)
		{
			InitializeSybaseCommon(cfg);
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Sybase.Ase.ByteArrayAsImage"));
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Sybase.Ase.StringAsText")); // do not localize
//			cfg.SetPMapper("StringAsNText", RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.SqlServer.Bdp.StringAsNText")); // do not localize
		}
*/


		private static void InitializeOracle(SqlDatabaseConfig cfg)
		{
			/*
			FColumnTypeForString := 'VARCHAR2(%d)';
			FColumnTypeForFloat := 'NUMBER';
			FColumnTypeForCurrency := 'NUMBER';

			fColumnTypeForSmallInt := 'NUMBER(5,0)'; // do not localize
			*/
			cfg.DropColumnTemplate = "ALTER TABLE <TableName> DROP COLUMN <ColumnName>"; // do not localize
			cfg.MaxIndexNameLength = 30;
			cfg.MaxDbIdentifierLength = 30;
			cfg.SetPMapper(StringLiterals.SystemString, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.StringAsVarChar2")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt32, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.Int32AsNumber")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt16, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.Int16AsNumber")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemByte, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.ByteAsSmallInt")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDouble, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.DoubleAsNumber")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemSingle, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.SingleAsSingle")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Oracle.ByteArrayAsBlob")); // do not localize
			cfg.SetPMapper("StringAsClob", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Oracle.StringAsClob")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Oracle.StringAsClob")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDateTime, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.DateTimeAsDate")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemTimeSpan, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.TimeSpanAsDate")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemBoolean, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.BooleanAsInteger")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.DecimalAsDecimal")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt64, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.Int64AsLongInteger")); // do not localize
			cfg.SetPMapper("StringAsNClob", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Oracle.StringAsNClob")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoInc, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Oracle.Int32AsAutoInc"));

			cfg.AddReservedWord("ACCESS"); // do not localize
			cfg.AddReservedWord("AUDIT"); // do not localize
			cfg.AddReservedWord("CLUSTER"); // do not localize
			cfg.AddReservedWord("COMMENT"); // do not localize
			cfg.AddReservedWord("COMPRESS"); // do not localize
			cfg.AddReservedWord("CONNECT"); // do not localize
			cfg.AddReservedWord("EXCLUSIVE"); // do not localize
			cfg.AddReservedWord("IDENTIFIED"); // do not localize
			cfg.AddReservedWord("IMMEDIATE"); // do not localize
			cfg.AddReservedWord("INCREMENT"); // do not localize
			cfg.AddReservedWord("INITIAL"); // do not localize
			cfg.AddReservedWord("INTERSECT"); // do not localize
			cfg.AddReservedWord("LOCK"); // do not localize
			cfg.AddReservedWord("LONG"); // do not localize
			cfg.AddReservedWord("MAXEXTENTS"); // do not localize
			cfg.AddReservedWord("MINUS"); // do not localize
			cfg.AddReservedWord("MLSLABEL"); // do not localize
			cfg.AddReservedWord("MODE"); // do not localize
			cfg.AddReservedWord("MODIFY"); // do not localize
			cfg.AddReservedWord("NOAUDIT"); // do not localize
			cfg.AddReservedWord("NOCOMPRESS"); // do not localize
			cfg.AddReservedWord("NOTFOUND"); // do not localize
			cfg.AddReservedWord("NOWAIT"); // do not localize
			cfg.AddReservedWord("NUMBER"); // do not localize
			cfg.AddReservedWord("OFFLINE"); // do not localize
			cfg.AddReservedWord("ONLINE"); // do not localize
			cfg.AddReservedWord("PCTFREE"); // do not localize
			cfg.AddReservedWord("PRIOR"); // do not localize
			cfg.AddReservedWord("PUBLIC"); // do not localize
			cfg.AddReservedWord("RAW"); // do not localize
			cfg.AddReservedWord("RENAME"); // do not localize
			cfg.AddReservedWord("RESOURCE"); // do not localize
			cfg.AddReservedWord("ROW"); // do not localize
			cfg.AddReservedWord("ROWID"); // do not localize
			cfg.AddReservedWord("ROWLABEL"); // do not localize
			cfg.AddReservedWord("ROWNUM"); // do not localize
			cfg.AddReservedWord("ROWS"); // do not localize
			cfg.AddReservedWord("SESSION"); // do not localize
			cfg.AddReservedWord("SHARE"); // do not localize
			cfg.AddReservedWord("SQLBUF"); // do not localize
			cfg.AddReservedWord("START"); // do not localize
			cfg.AddReservedWord("SUCCESSFUL"); // do not localize
			cfg.AddReservedWord("SYNONYM"); // do not localize
			cfg.AddReservedWord("SYSDATE"); // do not localize
			cfg.AddReservedWord("UID"); // do not localize
			cfg.AddReservedWord("VALIDATE"); // do not localize
			cfg.AddReservedWord("VALIDATION"); // do not localize
			cfg.AddReservedWord("VARCHAR2"); // do not localize
			cfg.AddReservedWord("WHENEVER"); // do not localize
		}

		private static void InitializeOracleMBCS(SqlDatabaseConfig cfg)
		{
			InitializeOracle(cfg);
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Oracle.StringAsNClob")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemString, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.StringAsNVarChar")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemChar, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.CharAsNChar")); // do not localize
		}

		private static void InitializeInformix(SqlDatabaseConfig cfg)
		{
				cfg.MaxIndexNameLength = 18;
				cfg.MaxDbIdentifierLength = 18;
				/*
				fColumnTypeForCurrency := 'MONEY';
				fColumnTypeForFloat := 'NUMERIC';
				fColumnTypeForDate := 'DATETIME YEAR TO DAY';
				fColumnTypeForTime := 'DATETIME HOUR TO FRACTION';
				fColumnTypeForDateTime := 'DATETIME YEAR TO FRACTION';
				*/
		}

		private static void InitializeNDataStoreCommon(SqlDatabaseConfig cfg)
		{
			cfg.QuoteSqlNames = true;
			cfg.SqlQuoteChar = "\"";
			cfg.MaxDbIdentifierLength = 128;
			cfg.MaxIndexNameLength = 128;

			cfg.SetPMapper(StringLiterals.SystemInt64, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.NDataStore.Int64AsBigInt")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemBoolean, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.NDataStore.BooleanAsBoolean")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemSingle, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.NDataStore.SingleAsFloat")); // do not localize
//			cfg.SetPMapper(StringLiterals.SystemString, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.NDataStore.StringAsVarChar")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemTimeSpan, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.TimeSpanAsTimeStamp")); // do not localize
			cfg.SetPMapper(StringLiterals.AutoInc, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.NDataStore.Int32AsAutoInc")); // do not localize
		}

		private static void InitializeNDataStoreBdp(SqlDatabaseConfig cfg)
		{
			InitializeNDataStoreCommon(cfg);
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.NDataStore.ByteArrayAsBdpVarBinary")); // do not localize
		}

		private static void InitializeNDataStore(SqlDatabaseConfig cfg)
		{
			InitializeNDataStoreCommon(cfg);
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceNDataStore, "Borland.Eco.Persistence.NDataStore.ByteArrayAsVarBinary")); // do not localize
		}

		private static void InitializeMySqlBdp(SqlDatabaseConfig cfg)
		{
			InitializeGenericANSISQL92(cfg);
            cfg.MaxDbIdentifierLength = 64;
            cfg.MaxIndexNameLength = 64;
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.MySql.ByteArrayAsBdpBlob")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemTimeSpan, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.TimeSpanAsDateTime")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt64, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.MySql.Int64AsBigInt")); // do not localize
			cfg.SetPMapper("StringAsText", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.MySql.StringAsLongText")); // do not localize
		}


		private static void InitializeDB2(SqlDatabaseConfig cfg)
		{
			cfg.MaxDbIdentifierLength = 18;
			cfg.MaxIndexNameLength = 18;
			cfg.FetchBlockSize = 127;
			cfg.DropColumnTemplate = "ALTER TABLE <TableName> DROP COLUMN <ColumnName>"; // do not localize
			cfg.SetPMapper(StringLiterals.SystemByteArr, RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Oracle.ByteArrayAsBlob")); // do not localize
			cfg.SetPMapper("StringAsClob", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Oracle.StringAsClob")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemDecimal, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Db2.DecimalAsDecimal")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemInt64, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Db2.Int64AsBigInt")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemSingle, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Db2.SingleAsFloat")); // do not localize
			cfg.SetPMapper("StringAsDBClob", RetrieveType(StringLiterals.PersistenceBdp, "Borland.Eco.Persistence.Db2.StringAsDbClob")); // do not localize
		}

		private static void InitializeDB2MBCS(SqlDatabaseConfig cfg)
		{
			InitializeDB2(cfg);
			cfg.SetPMapper(StringLiterals.SystemString, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.StringAsNVarChar")); // do not localize
			cfg.SetPMapper(StringLiterals.SystemChar, RetrieveType(StringLiterals.Persistence, "Borland.Eco.Persistence.Default.CharAsNChar")); // do not localize
		}

#endregion

		///<summary>
		///Used by designers to configure a persistencemapper's SqlDatabaseConfig.
		///Can also be invoked in runtime prior to opening the system.
		///</summary>
		public static void InitializeDbEngineSettings(DataBaseEngine engine, SqlDatabaseConfig sqlDatabaseConfig)
		{
			SetInitialValues(sqlDatabaseConfig);
			switch (engine)
			{
				case DataBaseEngine.SqlServerBdp:
					InitializeSQLServerBdp(sqlDatabaseConfig);
				break;

				case DataBaseEngine.SqlServerSqlClient:
					InitializeSQLServerSqlClient(sqlDatabaseConfig);
				break;

				case DataBaseEngine.SqlServer7Bdp:
					InitializeSQLServer7Bdp(sqlDatabaseConfig);
				break;

				case DataBaseEngine.SqlServer7SqlClient:
					InitializeSQLServer7SqlClient(sqlDatabaseConfig);
				break;

				case DataBaseEngine.SqlServerMBCSBdp:
					InitializeSQLServerMBCSBdp(sqlDatabaseConfig);
				break;

				case DataBaseEngine.SqlServerMBCSSqlClient:
					InitializeSQLServerMBCSSqlClient(sqlDatabaseConfig);
				break;

				case DataBaseEngine.SqlServer7MBCSBdp:
					InitializeSQLServer7MBCSBdp(sqlDatabaseConfig);
				break;

				case DataBaseEngine.SqlServer7MBCSSqlClient:
					InitializeSQLServer7MBCSSqlClient(sqlDatabaseConfig);
				break;

				case DataBaseEngine.GenericANSISQL92:
					InitializeGenericANSISQL92(sqlDatabaseConfig);
				break;

				case DataBaseEngine.InterbaseSqlDialect3:
					InitializeInterbaseSQLDialect3(sqlDatabaseConfig);
				break;

				case DataBaseEngine.InterbaseSqlDialect3MBCS:
					InitializeInterbaseSQLDialect3MBCS(sqlDatabaseConfig);
				break;

				case DataBaseEngine.InterbaseSqlDialect1:
					InitializeInterbaseSQLDialect1(sqlDatabaseConfig);
				break;

				case DataBaseEngine.DBISAM:
					InitializeDBISAM(sqlDatabaseConfig);
				break;

				case DataBaseEngine.Advantage:
					InitializeAdvantage(sqlDatabaseConfig);
				break;

				case DataBaseEngine.Oracle:
					InitializeOracle(sqlDatabaseConfig);
				break;

				case DataBaseEngine.OracleMBCS:
					InitializeOracleMBCS(sqlDatabaseConfig);
				break;

				case DataBaseEngine.Informix:
						InitializeInformix(sqlDatabaseConfig);
				break;

				case DataBaseEngine.DB2:
					InitializeDB2(sqlDatabaseConfig);
				break;

				case DataBaseEngine.DB2MBCS:
					InitializeDB2MBCS(sqlDatabaseConfig);
				break;
				
				case DataBaseEngine.SybaseBdp:
					InitializeSybaseBdp(sqlDatabaseConfig);
				break;
				
				case DataBaseEngine.SybaseCommon:
					InitializeSybaseCommon(sqlDatabaseConfig);
				break;

				case DataBaseEngine.AccessBdp:
					InitializeAccessBdp(sqlDatabaseConfig);
				break;

				case DataBaseEngine.AccessSqlClient:
					InitializeAccessSqlClient(sqlDatabaseConfig);
				break;

				case DataBaseEngine.NDataStore:
					InitializeNDataStore(sqlDatabaseConfig);
				break;

				case DataBaseEngine.NDataStoreBdp:
					InitializeNDataStoreBdp(sqlDatabaseConfig);
				break;

				case DataBaseEngine.MySqlBdp:
					InitializeMySqlBdp(sqlDatabaseConfig);
				break;

			}
		}
	}
}
